<?php
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/db.php';
require_login();
header('Content-Type: application/json');

$action = $_GET['action'] ?? $_POST['action'] ?? 'list';
$baseDir = realpath(__DIR__ . '/..');
$backupDir = $baseDir . DIRECTORY_SEPARATOR . 'backups';
if (!is_dir($backupDir)) {
    @mkdir($backupDir, 0775, true);
}

function ensure_admin_backup() { if (!has_role(['admin','super_admin'])) json_response(['message'=>'Forbidden'], 403); }

function list_backups($backupDir) {
    $out = [];
    if (is_dir($backupDir)) {
        $files = scandir($backupDir);
        foreach ($files as $f) {
            if ($f === '.' || $f === '..') continue;
            $full = $backupDir . DIRECTORY_SEPARATOR . $f;
            if (is_file($full)) {
                $out[] = [
                    'name' => $f,
                    'size' => filesize($full),
                    'mtime' => date('Y-m-d H:i:s', filemtime($full)),
                    'url' => url_for('backups/' . rawurlencode($f))
                ];
            }
        }
        // sort desc by mtime
        usort($out, function($a,$b){ return strcmp($b['mtime'],$a['mtime']); });
    }
    return $out;
}

switch ($action) {
    case 'list':
        json_response(['data' => list_backups($backupDir)]);
        break;

    case 'db_backup':
        ensure_admin_backup();
        if (!validate_csrf($_POST['csrf'] ?? '')) json_response(['message' => 'Invalid CSRF'], 400);
        // Build filename
        $fname = 'db_' . DB_NAME . '_' . date('Ymd_His') . '.sql';
        $fpath = $backupDir . DIRECTORY_SEPARATOR . $fname;

        // Try to locate mysqldump on Windows XAMPP or PATH
        $candidates = [
            'mysqldump',
            'C:\\xampp\\mysql\\bin\\mysqldump.exe',
            'C:\\Program Files\\MySQL\\MySQL Server 8.0\\bin\\mysqldump.exe',
        ];
        $mysqldump = null;
        foreach ($candidates as $cand) {
            $cmd = (stripos(PHP_OS, 'WIN') === 0) ? 'where ' . escapeshellarg($cand) : 'command -v ' . escapeshellarg($cand);
            $res = @shell_exec($cmd);
            if ($res) { $mysqldump = trim(explode("\n", $res)[0]); break; }
            if (is_file($cand)) { $mysqldump = $cand; break; }
        }
        if (!$mysqldump) {
            // fallback assume command available
            $mysqldump = 'mysqldump';
        }

        // Construct command with proper escaping
        $host = DB_HOST; $user = DB_USER; $pass = DB_PASS; $db = DB_NAME;
        $redirect = (stripos(PHP_OS, 'WIN') === 0) ? '>' : '>'; // same
        $cmd = '"' . $mysqldump . '" --single-transaction --quick --lock-tables=false '
             . '-h ' . escapeshellarg($host) . ' -u ' . escapeshellarg($user)
             . ($pass !== '' ? ' -p' . escapeshellarg($pass) : '') . ' '
             . escapeshellarg($db) . ' ' . $redirect . ' ' . escapeshellarg($fpath);

        $exitCode = null;
        if (function_exists('proc_open')) {
            $proc = proc_open($cmd, [1=>['pipe','w'], 2=>['pipe','w']], $pipes, $baseDir);
            if (is_resource($proc)) {
                $stdout = stream_get_contents($pipes[1]); fclose($pipes[1]);
                $stderr = stream_get_contents($pipes[2]); fclose($pipes[2]);
                $exitCode = proc_close($proc);
                if ($exitCode !== 0) {
                    // Clean partial file
                    if (file_exists($fpath) && filesize($fpath) === 0) @unlink($fpath);
                    json_response(['message' => 'mysqldump failed', 'error' => trim($stderr)], 500);
                }
            } else {
                json_response(['message' => 'Failed to start mysqldump process'], 500);
            }
        } else {
            // Fallback to shell_exec
            $out = @shell_exec($cmd . ' 2>&1');
            if (!file_exists($fpath) || filesize($fpath) === 0) {
                json_response(['message' => 'mysqldump failed', 'error' => trim((string)$out)], 500);
            }
        }

        json_response(['message' => 'DB backup created', 'url' => url_for('backups/' . rawurlencode($fname))]);
        break;

    case 'system_backup':
        ensure_admin_backup();
        if (!validate_csrf($_POST['csrf'] ?? '')) json_response(['message' => 'Invalid CSRF'], 400);
        if (!class_exists('ZipArchive')) {
            json_response(['message' => 'ZipArchive not available in PHP'], 500);
        }
        $fname = 'system_' . date('Ymd_His') . '.zip';
        $fpath = $backupDir . DIRECTORY_SEPARATOR . $fname;
        $zip = new ZipArchive();
        if ($zip->open($fpath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            json_response(['message' => 'Failed to create ZIP archive'], 500);
        }
        $root = $baseDir; // application root
        $exclude = [realpath($backupDir)];
        $it = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($root, FilesystemIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );
        foreach ($it as $file) {
            $full = $file->getPathname();
            // Exclude backups directory
            foreach ($exclude as $ex) {
                if ($ex && strpos(realpath($full), $ex) === 0) { continue 2; }
            }
            $localName = ltrim(str_replace($root, '', $full), DIRECTORY_SEPARATOR);
            if ($file->isDir()) {
                $zip->addEmptyDir($localName);
            } else {
                $zip->addFile($full, $localName);
            }
        }
        $zip->close();
        json_response(['message' => 'System backup created', 'url' => url_for('backups/' . rawurlencode($fname))]);
        break;

    default:
        json_response(['message' => 'Invalid action'], 400);
}
