<?php
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/db.php';
require_login();
header('Content-Type: application/json');
$mysqli = db();
$action = $_GET['action'] ?? $_POST['action'] ?? 'list_pieces';

function ensure_cutting_schema(mysqli $db){
  // store location for cut slabs
  @$db->query("CREATE TABLE IF NOT EXISTS inventory_moves (
    id INT AUTO_INCREMENT PRIMARY KEY,
    from_location_id INT NULL,
    to_location_id INT NULL,
    ref_type VARCHAR(50) NOT NULL,
    ref_id INT NOT NULL,
    quantity DECIMAL(12,3) NOT NULL DEFAULT 0,
    unit VARCHAR(20) NOT NULL DEFAULT 'sqft',
    notes VARCHAR(255) NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  ) ENGINE=InnoDB");

  @$db->query("CREATE TABLE IF NOT EXISTS cutting_jobs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    job_date DATE NOT NULL DEFAULT (CURRENT_DATE),
    source_type ENUM('raw','finished') NOT NULL,
    source_id INT NOT NULL,
    from_location_id INT NULL,
    to_location_id INT NULL,
    operator_name VARCHAR(150) NULL,
    notes VARCHAR(255) NULL,
    status ENUM('pending','completed','cancelled') NOT NULL DEFAULT 'completed',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  ) ENGINE=InnoDB");

  @$db->query("CREATE TABLE IF NOT EXISTS cut_pieces (
    id INT AUTO_INCREMENT PRIMARY KEY,
    job_id INT NOT NULL,
    parent_piece_id INT NULL,
    piece_no TINYINT NOT NULL,
    length_mm DECIMAL(12,2) NOT NULL,
    width_mm DECIMAL(12,2) NOT NULL,
    thickness_mm DECIMAL(12,2) NULL,
    area_sq_ft DECIMAL(12,3) NOT NULL,
    status ENUM('usable','unusable') NOT NULL DEFAULT 'usable',
    store_location_id INT NULL,
    is_available TINYINT(1) NOT NULL DEFAULT 1,
    reserved_invoice_item_id INT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    CONSTRAINT fk_cut_job FOREIGN KEY (job_id) REFERENCES cutting_jobs(id) ON DELETE CASCADE
  ) ENGINE=InnoDB");

  // Add parent_piece_id column if not exists (for re-cut linkage)
  @$db->query("ALTER TABLE cut_pieces ADD COLUMN IF NOT EXISTS parent_piece_id INT NULL AFTER job_id");
  // Add status and location columns if not exists
  @$db->query("ALTER TABLE cut_pieces ADD COLUMN IF NOT EXISTS status ENUM('usable','unusable') NOT NULL DEFAULT 'usable' AFTER area_sq_ft");
  @$db->query("ALTER TABLE cut_pieces ADD COLUMN IF NOT EXISTS store_location_id INT NULL AFTER status");

  // Ensure a default Cut Slab Warehouse exists
  $res = $db->query("SELECT id FROM store_locations WHERE name='Cut Slab Warehouse' LIMIT 1");
  if (!$res || !$res->fetch_assoc()) {
    @$db->query("INSERT INTO store_locations (name, notes, is_active) VALUES ('Cut Slab Warehouse', 'Auto-created for cutting workflow', 1)");
  }
}

function json_ok($data){ echo json_encode($data); exit; }
function json_err($msg,$code=400){ http_response_code($code); echo json_encode(['message'=>$msg]); exit; }

ensure_cutting_schema($mysqli);

switch($action){
  case 'list_sources':
    $type = $_GET['type'] ?? 'raw';
    $q = trim($_GET['q'] ?? '');
    if ($type === 'finished') {
      // Join raw_material_products for slab_no, purchase_orders for po_no, finishes for finish_name
      $sql = "SELECT fp.id, CONCAT('FP-', fp.id) AS code,
                     fp.final_length AS length_mm, fp.final_height AS width_mm, fp.thickness_mm AS thickness_mm,
                     rmp.slab_no AS slab_no, po.order_no AS po_no, f.name AS finish_name
              FROM final_products fp
              LEFT JOIN raw_material_products rmp ON rmp.id = fp.rmp_id
              LEFT JOIN purchase_orders po ON po.id = fp.po_id
              LEFT JOIN finishes f ON f.id = fp.finish_id";
      $where = '';
      if ($q !== '') { $where = " WHERE rmp.slab_no LIKE CONCAT('%', ?, '%')"; }
      $sql .= $where . " ORDER BY fp.id DESC LIMIT 200";
      if ($q !== '') { $stmt=$mysqli->prepare($sql); $stmt->bind_param('s',$q); $stmt->execute(); $res=$stmt->get_result(); }
      else { $res = $mysqli->query($sql); }
    } else if ($type === 'raw') {
      // Raw: join purchase_orders for po_no; slab_no is in raw_material_products
      $sql = "SELECT rmp.id, CONCAT('RM-', rmp.id) AS code,
                     rmp.length*304.8 AS length_mm, rmp.height*304.8 AS width_mm, NULL AS thickness_mm,
                     rmp.slab_no AS slab_no, po.order_no AS po_no, NULL AS finish_name
              FROM raw_material_products rmp
              LEFT JOIN purchase_orders po ON po.id = rmp.po_id";
      $where = '';
      if ($q !== '') { $where = " WHERE rmp.slab_no LIKE CONCAT('%', ?, '%')"; }
      $sql .= $where . " ORDER BY rmp.id DESC LIMIT 200";
      if ($q !== '') { $stmt=$mysqli->prepare($sql); $stmt->bind_param('s',$q); $stmt->execute(); $res=$stmt->get_result(); }
      else { $res = $mysqli->query($sql); }
    } else if ($type === 'cut_piece') {
      // From available cut pieces; include slab_no/po_no via source chain
      $sql = "SELECT cp.id, CONCAT('CP-', cp.id) AS code,
                     cp.length_mm AS length_mm, cp.width_mm AS width_mm, cp.thickness_mm AS thickness_mm,
                     COALESCE(rmp_fp.slab_no, rmp_raw.slab_no) AS slab_no,
                     COALESCE(po_fp.order_no, po_raw.order_no) AS po_no,
                     f.name AS finish_name
              FROM cut_pieces cp
              LEFT JOIN cutting_jobs cj ON cj.id = cp.job_id
              LEFT JOIN final_products fp ON (cj.source_type='finished' AND fp.id=cj.source_id)
              LEFT JOIN finishes f ON f.id = fp.finish_id
              LEFT JOIN raw_material_products rmp_fp ON rmp_fp.id = fp.rmp_id
              LEFT JOIN purchase_orders po_fp ON po_fp.id = fp.po_id
              LEFT JOIN raw_material_products rmp_raw ON (cj.source_type='raw' AND rmp_raw.id=cj.source_id)
              LEFT JOIN purchase_orders po_raw ON po_raw.id = rmp_raw.po_id
              WHERE cp.is_available=1";
      if ($q !== '') { $sql .= " AND COALESCE(rmp_fp.slab_no, rmp_raw.slab_no) LIKE CONCAT('%', ?, '%')"; }
      $sql .= " ORDER BY cp.id DESC LIMIT 200";
      if ($q !== '') { $stmt=$mysqli->prepare($sql); $stmt->bind_param('s',$q); $stmt->execute(); $res=$stmt->get_result(); }
      else { $res = $mysqli->query($sql); }
      $rows=[]; if($res){ while($r=$res->fetch_assoc()){ $rows[]=$r; } }
      json_ok(['data'=>$rows]);
      break;
    } else {
      json_err('Invalid source type', 400);
    }
    $rows=[]; if($res){ while($r=$res->fetch_assoc()){ $rows[]=$r; } }
    json_ok(['data'=>$rows]);
    break;

  case 'create_job':
    if (!validate_csrf($_POST['csrf'] ?? '')) json_err('Invalid CSRF', 400);
    $source_type = ($_POST['source_type'] ?? 'raw') === 'finished' ? 'finished' : 'raw';
    $source_id = (int)($_POST['source_id'] ?? 0);
    if(!$source_id) json_err('Missing source');
    $operator = trim($_POST['operator_name'] ?? '');
    $notes = trim($_POST['notes'] ?? '');
    // get cut slab warehouse id
    $loc = $mysqli->query("SELECT id FROM store_locations WHERE name='Cut Slab Warehouse' LIMIT 1");
    $cutLoc = ($loc && ($x=$loc->fetch_assoc())) ? (int)$x['id'] : null;

    // Try to read thickness from source when possible
    $thk = null;
    if ($source_type==='finished'){
      $q=$mysqli->query("SELECT thickness_mm FROM final_products WHERE id=".(int)$source_id." LIMIT 1");
      if($q && ($r=$q->fetch_assoc())) $thk = $r['thickness_mm'];
    } else if ($source_type==='cut_piece') {
      $q=$mysqli->query("SELECT thickness_mm FROM cut_pieces WHERE id=".(int)$source_id." LIMIT 1");
      if($q && ($r=$q->fetch_assoc())) $thk = $r['thickness_mm'];
    }

    $stmt = $mysqli->prepare("INSERT INTO cutting_jobs (job_date, source_type, source_id, from_location_id, to_location_id, operator_name, notes, status) VALUES (CURRENT_DATE,?,?,?,?,?,?, 'completed')");
    $fromLoc = null; // unknown for now
    $stmt->bind_param('siisss', $source_type, $source_id, $fromLoc, $cutLoc, $operator, $notes);
    if(!$stmt->execute()) json_err($stmt->error,500);
    $job_id = $mysqli->insert_id;

    // If re-cutting a cut piece, consume the source and log movement out
    $parent_piece_id = null;
    if ($source_type === 'cut_piece') {
      $parent_piece_id = $source_id;
      $px = $mysqli->query('SELECT area_sq_ft FROM cut_pieces WHERE id='.(int)$source_id.' LIMIT 1');
      $pArea = 0.0; if ($px && ($pr=$px->fetch_assoc())) $pArea = (float)$pr['area_sq_ft'];
      // mark parent as unavailable (consumed)
      @$mysqli->query('UPDATE cut_pieces SET is_available=0 WHERE id='.(int)$source_id);
      // movement out from Cut Slab Warehouse
      if ($cutLoc && $pArea>0) {
        $mv2 = $mysqli->prepare("INSERT INTO inventory_moves (from_location_id, to_location_id, ref_type, ref_id, quantity, unit, notes) VALUES (?, NULL, 'cutting_job', ?, ?, 'sqft', 'Cut piece consumed by re-cut')");
        $mv2->bind_param('iid', $cutLoc, $job_id, $pArea);
        @$mv2->execute();
      }
    }

    // pieces: prefer dynamic JSON, fallback to legacy two inputs
    $pieces = [];
    if (!empty($_POST['pieces_json'])) {
      $decoded = json_decode($_POST['pieces_json'], true);
      if (is_array($decoded)) {
        foreach ($decoded as $row) {
          $no = isset($row['piece_no']) ? (int)$row['piece_no'] : (count($pieces)+1);
          $L = isset($row['length_mm']) ? (float)$row['length_mm'] : 0;
          $W = isset($row['width_mm']) ? (float)$row['width_mm'] : 0;
          $pieces[] = [$no, $L, $W];
        }
      }
    }
    if (empty($pieces)) {
      $pieces = [
        [1, (float)($_POST['p1_length_mm'] ?? 0), (float)($_POST['p1_width_mm'] ?? 0)],
        [2, (float)($_POST['p2_length_mm'] ?? 0), (float)($_POST['p2_width_mm'] ?? 0)],
      ];
    }
    // Validate at least one piece is provided
    $validCount = 0; foreach($pieces as $pc){ if (($pc[1]??0)>0 && ($pc[2]??0)>0) { $validCount++; } }
    if ($validCount === 0) { json_err('Please add at least one piece with valid Length and Width (mm).', 400); }

    $ins = $mysqli->prepare("INSERT INTO cut_pieces (job_id, parent_piece_id, piece_no, length_mm, width_mm, thickness_mm, area_sq_ft, status, store_location_id) VALUES (?,?,?,?,?,?,?,?,?)");
    foreach($pieces as $pc){
      [$no,$L,$W] = $pc;
      if ($L>0 && $W>0){
        $area_sqft = ($L/304.8)*($W/304.8);
        // parent_piece_id is set only for re-cut, else NULL
        $status = 'usable';
        $locId = $cutLoc ?: null;
        if ($parent_piece_id) {
          $ins->bind_param('iiidddddi', $job_id, $parent_piece_id, $no, $L, $W, $thk, $area_sqft, $status, $locId);
        } else {
          $null = null;
          $ins->bind_param('iiidddddi', $job_id, $null, $no, $L, $W, $thk, $area_sqft, $status, $locId);
        }
        if(!$ins->execute()) json_err($ins->error,500);
      }
    }

    // movement logs (simple)
    $total_area = 0.0;
    $rs = $mysqli->query("SELECT SUM(area_sq_ft) s FROM cut_pieces WHERE job_id=".(int)$job_id);
    if($rs && ($r=$rs->fetch_assoc())) $total_area = (float)$r['s'];
    if ($cutLoc) {
      $mv = $mysqli->prepare("INSERT INTO inventory_moves (from_location_id, to_location_id, ref_type, ref_id, quantity, unit, notes) VALUES (NULL, ?, 'cutting_job', ?, ?, 'sqft', 'Cut pieces created')");
      $mv->bind_param('iid', $cutLoc, $job_id, $total_area);
      @$mv->execute();
    }

    json_ok(['message'=>'Cutting job created','job_id'=>$job_id]);
    break;

  case 'list_jobs':
    $res = $mysqli->query("SELECT id, job_date, source_type, source_id, operator_name, status FROM cutting_jobs ORDER BY id DESC LIMIT 200");
    $rows=[]; if($res){ while($r=$res->fetch_assoc()){ $rows[]=$r; } }
    json_ok(['data'=>$rows]);
    break;

  case 'delete_job':
    if (!validate_csrf($_POST['csrf'] ?? '')) json_err('Invalid CSRF', 400);
    $id = (int)($_POST['id'] ?? 0); if(!$id) json_err('Missing id');
    // Read job first
    $job = $mysqli->query('SELECT id, source_type, source_id FROM cutting_jobs WHERE id='.(int)$id.' LIMIT 1');
    $row = $job ? $job->fetch_assoc() : null; if(!$row) json_err('Not found', 404);
    // If re-cut, restore parent cut piece availability
    if (($row['source_type'] ?? '') === 'cut_piece') {
      @$mysqli->query('UPDATE cut_pieces SET is_available=1 WHERE id='.(int)$row['source_id']);
    }
    // Clean inventory moves for this job
    @$mysqli->query("DELETE FROM inventory_moves WHERE ref_type='cutting_job' AND ref_id=".(int)$id);
    // Delete job (FK will cascade delete cut_pieces)
    $mysqli->query('DELETE FROM cutting_jobs WHERE id='.(int)$id);
    json_ok(['message'=>'Deleted']);
    break;

  case 'list_pieces':
    $res = $mysqli->query("SELECT cp.id, cp.job_id, cp.piece_no, cp.length_mm, cp.width_mm, cp.thickness_mm, cp.area_sq_ft, cp.is_available FROM cut_pieces cp ORDER BY cp.id DESC LIMIT 500");
    $rows=[]; if($res){ while($r=$res->fetch_assoc()){ $rows[]=$r; } }
    json_ok(['data'=>$rows]);
    break;

  case 'list_available_pieces':
    $res = $mysqli->query("SELECT id, job_id, piece_no, length_mm, width_mm, thickness_mm, area_sq_ft FROM cut_pieces WHERE is_available=1 AND status='usable' ORDER BY id DESC LIMIT 500");
    $rows=[]; if($res){ while($r=$res->fetch_assoc()){ $rows[]=$r; } }
    json_ok(['data'=>$rows]);
    break;

  case 'reserve_piece':
    if (!validate_csrf($_POST['csrf'] ?? '')) json_err('Invalid CSRF', 400);
    $id = (int)($_POST['id'] ?? 0); if(!$id) json_err('Missing id');
    $invoice_item_id = isset($_POST['invoice_item_id']) ? (int)$_POST['invoice_item_id'] : null;
    $stmt = $mysqli->prepare('UPDATE cut_pieces SET is_available=0, reserved_invoice_item_id = ? WHERE id=? AND is_available=1');
    $stmt->bind_param('ii', $invoice_item_id, $id);
    if(!$stmt->execute() || !$stmt->affected_rows) json_err('Unable to reserve', 400);
    json_ok(['message'=>'Reserved']);
    break;

  case 'unreserve_piece':
    if (!validate_csrf($_POST['csrf'] ?? '')) json_err('Invalid CSRF', 400);
    $id = (int)($_POST['id'] ?? 0); if(!$id) json_err('Missing id');
    $stmt = $mysqli->prepare('UPDATE cut_pieces SET is_available=1, reserved_invoice_item_id = NULL WHERE id=?');
    $stmt->bind_param('i', $id);
    if(!$stmt->execute()) json_err('Unable to unreserve', 400);
    json_ok(['message'=>'Unreserved']);
    break;

  default:
    json_err('Invalid action',400);
}
