-- Database: granite_warehouse

CREATE DATABASE IF NOT EXISTS granite_warehouse CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE granite_warehouse;

-- Users
CREATE TABLE IF NOT EXISTS users (
  id INT AUTO_INCREMENT PRIMARY KEY,
  username VARCHAR(100) NOT NULL UNIQUE,
  full_name VARCHAR(150) NULL,
  role ENUM('super_admin','admin','sales_user') NOT NULL DEFAULT 'sales_user',
  is_active TINYINT(1) NOT NULL DEFAULT 1,
  password_hash VARCHAR(255) NOT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- ==========================================
-- Production Inventory (custom slabs/items)
-- ==========================================
CREATE TABLE IF NOT EXISTS production_qualities (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(100) NOT NULL UNIQUE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS production_items (
  id INT AUTO_INCREMENT PRIMARY KEY,
  type_id INT NOT NULL,
  quality_id INT NULL,
  name VARCHAR(150) NULL,
  color_id INT NULL,
  finish_id INT NULL,
  length DECIMAL(10,2) NOT NULL,
  width DECIMAL(10,2) NOT NULL,
  thickness DECIMAL(10,2) NOT NULL,
  quantity INT NOT NULL DEFAULT 1,
  availability ENUM('available','reserved','sold') NOT NULL DEFAULT 'available',
  notes VARCHAR(255) NULL,
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  CONSTRAINT fk_prod_item_type FOREIGN KEY (type_id) REFERENCES product_types(id) ON DELETE RESTRICT,
  CONSTRAINT fk_prod_item_quality FOREIGN KEY (quality_id) REFERENCES production_qualities(id) ON DELETE SET NULL,
  CONSTRAINT fk_prod_item_color FOREIGN KEY (color_id) REFERENCES colors(id) ON DELETE SET NULL,
  CONSTRAINT fk_prod_item_finish FOREIGN KEY (finish_id) REFERENCES finishes(id) ON DELETE SET NULL
) ENGINE=InnoDB;

-- Link production items to invoice items
ALTER TABLE invoice_items
  ADD COLUMN IF NOT EXISTS production_item_id INT NULL AFTER final_product_id;
ALTER TABLE invoice_items
  ADD CONSTRAINT IF NOT EXISTS fk_inv_item_prod FOREIGN KEY (production_item_id) REFERENCES production_items(id) ON DELETE SET NULL;

-- Link users to branches (nullable) [moved after branches table creation]

-- ==========================================
-- Migrations: Polishing and Final Products
-- ==========================================

-- Enrich polishing_jobs with optional metadata
ALTER TABLE polishing_jobs
  ADD COLUMN IF NOT EXISTS polishing_type VARCHAR(100) NULL AFTER notes,
  ADD COLUMN IF NOT EXISTS rate_per_sqft DECIMAL(12,2) NULL AFTER polishing_type,
  ADD COLUMN IF NOT EXISTS direct_cost DECIMAL(12,2) NULL AFTER rate_per_sqft;

-- Denormalize key fields into final_products for easier reporting
ALTER TABLE final_products
  ADD COLUMN IF NOT EXISTS pre_height DECIMAL(12,3) NULL AFTER polishing_job_id,
  ADD COLUMN IF NOT EXISTS pre_length DECIMAL(12,3) NULL AFTER pre_height,
  ADD COLUMN IF NOT EXISTS polisher_id INT NULL AFTER finish_id,
  ADD COLUMN IF NOT EXISTS polisher_name VARCHAR(150) NULL AFTER polisher_id,
  ADD COLUMN IF NOT EXISTS completed_at DATETIME NULL AFTER ready_at;

-- (moved below after employees table creation)

-- Raw material products (per-slab/items) imported from supplier CSV
CREATE TABLE IF NOT EXISTS raw_material_products (
  id INT AUTO_INCREMENT PRIMARY KEY,
  po_id INT NOT NULL,
  slab_no VARCHAR(50) NULL,
  height DECIMAL(12,3) NULL,
  length DECIMAL(12,3) NULL,
  sq_ft DECIMAL(12,3) NULL,
  sq_m2 DECIMAL(12,3) NULL,
  quality VARCHAR(50) NULL,
  price DECIMAL(12,3) NULL,
  value DECIMAL(12,3) NULL,
  length_remark VARCHAR(100) NULL,
  status ENUM('received','in_polishing','polished','finished','sold') NOT NULL DEFAULT 'received',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_rmp_po FOREIGN KEY (po_id) REFERENCES purchase_orders(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Import Raw Materials
CREATE TABLE IF NOT EXISTS imports (
  id INT AUTO_INCREMENT PRIMARY KEY,
  po_id INT NOT NULL,
  import_date DATE NOT NULL,
  reference_no VARCHAR(100) NULL,
  transporter VARCHAR(150) NULL,
  vehicle_no VARCHAR(50) NULL,
  warehouse VARCHAR(150) NULL,
  notes VARCHAR(255) NULL,
  attachment_path VARCHAR(255) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_import_po FOREIGN KEY (po_id) REFERENCES purchase_orders(id) ON DELETE RESTRICT
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS import_items (
  id INT AUTO_INCREMENT PRIMARY KEY,
  import_id INT NOT NULL,
  po_item_id INT NOT NULL,
  material_id INT NULL,
  qty_received DECIMAL(12,3) NOT NULL DEFAULT 0,
  batch_no VARCHAR(100) NULL,
  remarks VARCHAR(255) NULL,
  CONSTRAINT fk_import_item_import FOREIGN KEY (import_id) REFERENCES imports(id) ON DELETE CASCADE,
  CONSTRAINT fk_import_item_po_item FOREIGN KEY (po_item_id) REFERENCES purchase_order_items(id) ON DELETE RESTRICT
) ENGINE=InnoDB;

-- Allowances (monthly, employee-wise)
CREATE TABLE IF NOT EXISTS salary_allowances (
  id INT AUTO_INCREMENT PRIMARY KEY,
  employee_id INT NOT NULL,
  all_date DATE NOT NULL,
  amount DECIMAL(12,2) NOT NULL,
  reason VARCHAR(255) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_salary_all_emp FOREIGN KEY (employee_id) REFERENCES employees(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Product master data
CREATE TABLE IF NOT EXISTS product_types (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(100) NOT NULL UNIQUE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS finishes (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(100) NOT NULL UNIQUE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS colors (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(100) NOT NULL UNIQUE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Materials / Products
CREATE TABLE IF NOT EXISTS materials (
  id INT AUTO_INCREMENT PRIMARY KEY,
  material_code VARCHAR(30) NULL UNIQUE,
  name VARCHAR(150) NOT NULL,
  type_id INT NULL,
  finish_id INT NULL,
  color_id INT NULL,
  length_mm DECIMAL(10,2) NULL,
  width_mm DECIMAL(10,2) NULL,
  thickness_mm DECIMAL(10,2) NULL,
  notes TEXT NULL,
  is_active TINYINT(1) NOT NULL DEFAULT 1,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_material_type FOREIGN KEY (type_id) REFERENCES product_types(id) ON DELETE SET NULL ON UPDATE CASCADE,
  CONSTRAINT fk_material_finish FOREIGN KEY (finish_id) REFERENCES finishes(id) ON DELETE SET NULL ON UPDATE CASCADE,
  CONSTRAINT fk_material_color FOREIGN KEY (color_id) REFERENCES colors(id) ON DELETE SET NULL ON UPDATE CASCADE
) ENGINE=InnoDB;

-- Suppliers
CREATE TABLE IF NOT EXISTS suppliers (
  id INT AUTO_INCREMENT PRIMARY KEY,
  supplier_code VARCHAR(30) NULL UNIQUE,
  name VARCHAR(150) NOT NULL,
  contact_person VARCHAR(150) NULL,
  email VARCHAR(150) NULL,
  phone VARCHAR(50) NULL,
  address TEXT NULL,
  country VARCHAR(100) NULL,
  tax_id VARCHAR(30) NULL,
  payment_terms VARCHAR(50) NULL,
  currency VARCHAR(10) NULL,
  is_active TINYINT(1) NOT NULL DEFAULT 1,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Purchase Orders
CREATE TABLE IF NOT EXISTS purchase_orders (
  id INT AUTO_INCREMENT PRIMARY KEY,
  order_no VARCHAR(50) NOT NULL UNIQUE,
  supplier_id INT NOT NULL,
  order_date DATE NOT NULL,
  expected_date DATE NULL,
  status ENUM('pending','approved','received','cancelled') NOT NULL DEFAULT 'pending',
  ship_mode VARCHAR(50) NULL,
  delivery_term VARCHAR(50) NULL,
  payment_term VARCHAR(50) NULL,
  supplier_contact VARCHAR(150) NULL,
  supplier_email VARCHAR(150) NULL,
  notes TEXT NULL,
  total_amount DECIMAL(12,2) NOT NULL DEFAULT 0,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_po_supplier FOREIGN KEY (supplier_id) REFERENCES suppliers(id) ON DELETE RESTRICT ON UPDATE CASCADE
) ENGINE=InnoDB;

-- Purchase Order Items
CREATE TABLE IF NOT EXISTS purchase_order_items (
  id INT AUTO_INCREMENT PRIMARY KEY,
  po_id INT NOT NULL,
  line_no INT NOT NULL,
  description TEXT NOT NULL,
  hs_code VARCHAR(50) NULL,
  quantity DECIMAL(12,3) NOT NULL DEFAULT 0,
  unit VARCHAR(20) NULL,
  unit_price DECIMAL(12,3) NOT NULL DEFAULT 0,
  line_total DECIMAL(12,2) NOT NULL DEFAULT 0,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_poi_po FOREIGN KEY (po_id) REFERENCES purchase_orders(id) ON DELETE CASCADE ON UPDATE CASCADE
) ENGINE=InnoDB;

-- Ship Modes (PO master)
CREATE TABLE IF NOT EXISTS ship_modes (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(100) NOT NULL UNIQUE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Delivery Terms (PO master)
CREATE TABLE IF NOT EXISTS delivery_terms (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(100) NOT NULL UNIQUE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Payment Terms (PO master)
CREATE TABLE IF NOT EXISTS payment_terms (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(100) NOT NULL UNIQUE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Optional: Seed an initial super admin user (password must be set by app)
-- You can generate a password hash via PHP: password_hash('YourPassword', PASSWORD_DEFAULT)
-- INSERT INTO users (username, full_name, role, is_active, password_hash) VALUES ('superadmin', 'Super Admin', 'super_admin', 1, '<paste_hash_here>');

-- Shop Settings (single row)
CREATE TABLE IF NOT EXISTS shop_settings (
  id INT AUTO_INCREMENT PRIMARY KEY,
  company_name VARCHAR(150) NOT NULL DEFAULT 'Granite Warehouse',
  address_line VARCHAR(255) NOT NULL DEFAULT 'Warehouse Address Line',
  city_country VARCHAR(150) NOT NULL DEFAULT 'City, Country',
  company_email VARCHAR(150) NOT NULL DEFAULT '',
  phone VARCHAR(50) NOT NULL DEFAULT '-',
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Seed default settings if empty
INSERT INTO shop_settings (company_name, address_line, city_country, company_email, phone)
SELECT 'Granite Warehouse', 'Warehouse Address Line', 'City, Country', '', '-'
WHERE NOT EXISTS (SELECT 1 FROM shop_settings);

-- Migrations for Purchase Orders status workflow
-- Extend status values and add status date columns if not present
ALTER TABLE purchase_orders
  MODIFY COLUMN status ENUM('pending','confirmed','shipped','received','cancelled') NOT NULL DEFAULT 'pending';

ALTER TABLE purchase_orders
  ADD COLUMN IF NOT EXISTS confirmed_at DATE NULL AFTER expected_date,
  ADD COLUMN IF NOT EXISTS shipped_at DATE NULL AFTER confirmed_at,
  ADD COLUMN IF NOT EXISTS received_at DATE NULL AFTER shipped_at,
  ADD COLUMN IF NOT EXISTS cancelled_at DATE NULL AFTER received_at;

-- Financial Management: Expenditure Categories
CREATE TABLE IF NOT EXISTS expenditure_categories (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(100) NOT NULL UNIQUE,
  is_active TINYINT(1) NOT NULL DEFAULT 1,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Financial Management: Expenditures
CREATE TABLE IF NOT EXISTS expenditures (
  id INT AUTO_INCREMENT PRIMARY KEY,
  category_id INT NOT NULL,
  expense_date DATE NOT NULL,
  description VARCHAR(255) NOT NULL,
  amount DECIMAL(12,2) NOT NULL,
  payment_method VARCHAR(50) NOT NULL,
  vendor VARCHAR(150) DEFAULT NULL,
  invoice_no VARCHAR(100) DEFAULT NULL,
  receipt_path VARCHAR(255) DEFAULT NULL,
  is_recurring TINYINT(1) NOT NULL DEFAULT 0,
  notes TEXT NULL,
  created_by INT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_expenditures_category FOREIGN KEY (category_id) REFERENCES expenditure_categories(id) ON DELETE RESTRICT
) ENGINE=InnoDB;

-- Seed a few default categories if table empty
INSERT INTO expenditure_categories (name)
SELECT * FROM (SELECT 'Utilities' UNION SELECT 'Rent' UNION SELECT 'Transport' UNION SELECT 'Supplies' UNION SELECT 'Misc') AS tmp
WHERE NOT EXISTS (SELECT 1 FROM expenditure_categories);

-- Salary Management (basic placeholders)
CREATE TABLE IF NOT EXISTS employees (
  id INT AUTO_INCREMENT PRIMARY KEY,
  emp_code VARCHAR(50) UNIQUE,
  full_name VARCHAR(150) NOT NULL,
  phone VARCHAR(50) NULL,
  email VARCHAR(120) NULL,
  base_salary DECIMAL(12,2) DEFAULT 0,
  is_active TINYINT(1) NOT NULL DEFAULT 1,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS salary_advances (
  id INT AUTO_INCREMENT PRIMARY KEY,
  employee_id INT NOT NULL,
  adv_date DATE NOT NULL,
  amount DECIMAL(12,2) NOT NULL,
  notes VARCHAR(255) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_salary_adv_emp FOREIGN KEY (employee_id) REFERENCES employees(id) ON DELETE CASCADE
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS salary_deductions (
  id INT AUTO_INCREMENT PRIMARY KEY,
  employee_id INT NOT NULL,
  ded_date DATE NOT NULL,
  amount DECIMAL(12,2) NOT NULL,
  reason VARCHAR(255) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_salary_ded_emp FOREIGN KEY (employee_id) REFERENCES employees(id) ON DELETE CASCADE
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS salary_payroll (
  id INT AUTO_INCREMENT PRIMARY KEY,
  employee_id INT NOT NULL,
  period_month TINYINT NOT NULL,
  period_year SMALLINT NOT NULL,
  gross_salary DECIMAL(12,2) NOT NULL DEFAULT 0,
  total_incentives DECIMAL(12,2) NOT NULL DEFAULT 0,
  total_allowances DECIMAL(12,2) NOT NULL DEFAULT 0,
  total_advances DECIMAL(12,2) NOT NULL DEFAULT 0,
  total_deductions DECIMAL(12,2) NOT NULL DEFAULT 0,
  net_salary DECIMAL(12,2) NOT NULL DEFAULT 0,
  paid_on DATE NULL,
  notes VARCHAR(255) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_payroll_emp FOREIGN KEY (employee_id) REFERENCES employees(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Incentives (monthly, employee-wise)
CREATE TABLE IF NOT EXISTS salary_incentives (
  id INT AUTO_INCREMENT PRIMARY KEY,
  employee_id INT NOT NULL,
  inc_date DATE NOT NULL,
  amount DECIMAL(12,2) NOT NULL,
  reason VARCHAR(255) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_salary_inc_emp FOREIGN KEY (employee_id) REFERENCES employees(id) ON DELETE CASCADE
) ENGINE=InnoDB;
-- Polishing workflow jobs
CREATE TABLE IF NOT EXISTS polishing_jobs (
  id INT AUTO_INCREMENT PRIMARY KEY,
  rmp_id INT NOT NULL,
  po_id INT NOT NULL,
  pre_height DECIMAL(12,3) NULL,
  pre_length DECIMAL(12,3) NULL,
  worker_name VARCHAR(150) NULL,
  worker_id INT NULL,
  notes VARCHAR(255) NULL,
  status ENUM('pending','in_progress','completed','cancelled') NOT NULL DEFAULT 'in_progress',
  start_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  end_at DATETIME NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_polish_rmp FOREIGN KEY (rmp_id) REFERENCES raw_material_products(id) ON DELETE CASCADE,
  CONSTRAINT fk_polish_po FOREIGN KEY (po_id) REFERENCES purchase_orders(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Finalized products after polishing
CREATE TABLE IF NOT EXISTS final_products (
  id INT AUTO_INCREMENT PRIMARY KEY,
  rmp_id INT NOT NULL,
  po_id INT NOT NULL,
  polishing_job_id INT NOT NULL,
  final_height DECIMAL(12,3) NOT NULL,
  final_length DECIMAL(12,3) NOT NULL,
  final_sq_ft DECIMAL(12,3) NULL,
  final_sq_m2 DECIMAL(12,3) NULL,
  finish_id INT NULL,
  quality VARCHAR(50) NULL,
  sale_status ENUM('available','reserved','sold') NOT NULL DEFAULT 'available',
  ready_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_final_rmp FOREIGN KEY (rmp_id) REFERENCES raw_material_products(id) ON DELETE CASCADE,
  CONSTRAINT fk_final_po FOREIGN KEY (po_id) REFERENCES purchase_orders(id) ON DELETE CASCADE,
  CONSTRAINT fk_final_polish FOREIGN KEY (polishing_job_id) REFERENCES polishing_jobs(id) ON DELETE CASCADE,
  CONSTRAINT fk_final_finish FOREIGN KEY (finish_id) REFERENCES finishes(id) ON DELETE SET NULL
) ENGINE=InnoDB;

-- Branches (for assigning finished products)
CREATE TABLE IF NOT EXISTS branches (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(150) NOT NULL UNIQUE,
  address VARCHAR(255) NULL,
  phone VARCHAR(50) NULL,
  is_active TINYINT(1) NOT NULL DEFAULT 1,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Store Locations (managed in Settings)
CREATE TABLE IF NOT EXISTS store_locations (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(150) NOT NULL UNIQUE,
  notes VARCHAR(255) NULL,
  is_active TINYINT(1) NOT NULL DEFAULT 1,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Link store locations to branches
ALTER TABLE store_locations
  ADD COLUMN branch_id INT NULL AFTER id;
ALTER TABLE store_locations
  ADD CONSTRAINT fk_store_branch FOREIGN KEY (branch_id) REFERENCES branches(id) ON DELETE SET NULL;

-- Extend final_products with assignment and classification fields
ALTER TABLE final_products
  ADD COLUMN IF NOT EXISTS branch_id INT NULL AFTER sale_status,
  ADD COLUMN IF NOT EXISTS store_location_id INT NULL AFTER branch_id,
  ADD COLUMN IF NOT EXISTS type_id INT NULL AFTER finish_id,
  ADD COLUMN IF NOT EXISTS color_id INT NULL AFTER type_id;

-- Foreign keys for new columns
ALTER TABLE final_products
  ADD CONSTRAINT fk_final_branch FOREIGN KEY (branch_id) REFERENCES branches(id) ON DELETE SET NULL,
  ADD CONSTRAINT fk_final_store FOREIGN KEY (store_location_id) REFERENCES store_locations(id) ON DELETE SET NULL,
  ADD CONSTRAINT fk_final_type FOREIGN KEY (type_id) REFERENCES product_types(id) ON DELETE SET NULL,
  ADD CONSTRAINT fk_final_color FOREIGN KEY (color_id) REFERENCES colors(id) ON DELETE SET NULL;

-- Sales: customers
CREATE TABLE IF NOT EXISTS customers (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(150) NOT NULL,
  phone VARCHAR(50) NULL,
  email VARCHAR(150) NULL,
  address VARCHAR(255) NULL,
  gst_no VARCHAR(50) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Extend customers with additional profile fields
ALTER TABLE customers
  ADD COLUMN IF NOT EXISTS company_name VARCHAR(150) NULL AFTER name,
  ADD COLUMN IF NOT EXISTS city VARCHAR(100) NULL AFTER address,
  ADD COLUMN IF NOT EXISTS state VARCHAR(100) NULL AFTER city,
  ADD COLUMN IF NOT EXISTS country VARCHAR(100) NULL AFTER state;

-- Sales: invoices (currency LKR)
CREATE TABLE IF NOT EXISTS invoices (
  id INT AUTO_INCREMENT PRIMARY KEY,
  invoice_no VARCHAR(50) NOT NULL UNIQUE,
  customer_id INT NOT NULL,
  invoice_date DATE NOT NULL,
  status ENUM('draft','issued','paid','cancelled') NOT NULL DEFAULT 'draft',
  payment_term VARCHAR(20) NULL,
  subtotal DECIMAL(14,2) NOT NULL DEFAULT 0,
  tax_percent DECIMAL(5,2) NOT NULL DEFAULT 0,
  tax_amount DECIMAL(14,2) NOT NULL DEFAULT 0,
  transport_charges DECIMAL(12,2) NOT NULL DEFAULT 0,
  discount_percent DECIMAL(6,3) NULL,
  discount DECIMAL(14,2) NOT NULL DEFAULT 0,
  grand_total DECIMAL(14,2) NOT NULL DEFAULT 0,
  currency CHAR(3) NOT NULL DEFAULT 'LKR',
  notes VARCHAR(255) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_inv_cust FOREIGN KEY (customer_id) REFERENCES customers(id) ON DELETE RESTRICT
) ENGINE=InnoDB;

-- Sales: invoice items
CREATE TABLE IF NOT EXISTS invoice_items (
  id INT AUTO_INCREMENT PRIMARY KEY,
  invoice_id INT NOT NULL,
  final_product_id INT NULL,
  cut_piece_id INT NULL,
  production_item_id INT NULL,
  description VARCHAR(255) NOT NULL,
  qty DECIMAL(12,3) NOT NULL DEFAULT 1,
  unit_price DECIMAL(14,2) NOT NULL DEFAULT 0,
  line_total DECIMAL(14,2) NOT NULL DEFAULT 0,
  CONSTRAINT fk_inv_item_inv FOREIGN KEY (invoice_id) REFERENCES invoices(id) ON DELETE CASCADE,
  CONSTRAINT fk_inv_item_fp FOREIGN KEY (final_product_id) REFERENCES final_products(id) ON DELETE SET NULL,
  CONSTRAINT fk_inv_item_cut FOREIGN KEY (cut_piece_id) REFERENCES cut_pieces(id) ON DELETE SET NULL,
  CONSTRAINT fk_inv_item_prod FOREIGN KEY (production_item_id) REFERENCES production_items(id) ON DELETE SET NULL
) ENGINE=InnoDB;

-- Sales: payments
CREATE TABLE IF NOT EXISTS payments (
  id INT AUTO_INCREMENT PRIMARY KEY,
  invoice_id INT NOT NULL,
  pay_date DATE NOT NULL,
  method VARCHAR(50) NOT NULL,
  amount DECIMAL(14,2) NOT NULL,
  reference_no VARCHAR(100) NULL,
  notes VARCHAR(255) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_pay_inv FOREIGN KEY (invoice_id) REFERENCES invoices(id) ON DELETE CASCADE
) ENGINE=InnoDB;
